/** @file   aicontroller.cpp
 * @brief   Implementation of AIController - class.
 * @version $Revision: 1.5 $
 * @author  Tomi Lamminsaari
 */
 
#include "aicontroller.h"
#include "www_map.h"
#include "gameobject.h"
#include "utils.h"
#include "warglobals.h"
using namespace eng2d;
using std::vector;

namespace WeWantWar {


/** Constructor.
 */
AIController::AIController( GameObject* pAlien ) :
  BaseController( pAlien ),
  m_visualRange( 700 ),
  m_attackRange( 40 ),
  m_pTarget( 0 ),
  m_flags( 0 ),
  m_updateCounter( 0 ),
  m_targetSeen( false ),
  m_stuck( false )
{
}



/** Constructor.
 */
AIController::AIController( GameObject* pAlien, GameObject* pTarget ) :
  BaseController( pAlien ),
  
  m_pTarget( pTarget ),
  m_visualRange( 700 ),
  m_attackRange( 40 ),
  m_flags( 0 ),
  m_updateCounter( 0 ),
  m_targetSeen( false ),
  m_stuck( false )
{
}



/** Destructor.
 */
AIController::~AIController()
{
}



/** Sets the target object
 */
void AIController::setTarget( GameObject* pTarget )
{
  m_pTarget = pTarget;
}



/** Sets the behaviour flags.
 */
void AIController::setFlags( BFlag f )
{
  m_flags = f;
}



/** Updates the ai-controller.
 */
void AIController::update()
{
  BaseController::m_counter -= 1;
  if ( BaseController::m_counter >= 0 ) {
    return;
  }
  
  m_pTarget = WarGlobals::pObjTable->pPrimaryTarget;
  bool seeTarget = this->canSeeTarget();
  // Check if there is the primary object
  if ( WarGlobals::pObjTable->pPrimaryTarget != 0 ) {
    m_pTarget = WarGlobals::pObjTable->pPrimaryTarget;
    if ( m_pTarget != WarGlobals::pObjTable->pPlayer ) {
      // Primary target is other than player. If we cannot see the target
      // then we set the player to be our target.
      if ( seeTarget == false ) {
        m_pTarget = WarGlobals::pObjTable->pPlayer;
        seeTarget = this->canSeeTarget();
      }
    }
  }
  
  if ( m_targetSeen == false && m_flags & WAIT_TARGET != 0 ) {
    // We've not seen our target yet and we should stay put until
    // we see him.
    this->reset();
    m_targetSeen = this->canSeeTarget();
    this->setCounter( 10 + rand() % 10 );
    return;
  }
  
  // Every 10th time we store our position.
  m_updateCounter += 1;
  if ( m_updateCounter > 9 ) {
    m_updateCounter = 0;
    Vec2D tmpPos( BaseController::m_pObject->position() );
    tmpPos -= m_oldPos;
    m_stuck = false;
    if ( fabs( tmpPos.vx ) < 12 && fabs( tmpPos.vy ) < 12 ) {
      // We're stuck.
      m_stuck = true;
    }
    m_oldPos = BaseController::m_pObject->position();
  }
  
  
  // Call the actual behaviour logic.
  float d = Utils::objectDistance( BaseController::m_pObject, m_pTarget );
  this->updateAI( seeTarget, d );
}



/** Returns the flags
 */
AIController::BFlag AIController::getFlags() const
{
  return m_flags;
}



/** Checks if the alien can see the player.
 */
bool AIController::canSeeTarget() const
{
  if ( m_pTarget == 0 ) {
    return false;
  }

  Vec2D targetPos( m_pTarget->position() );

  // Create a directionvector from the alien to target.
  Vec2D dirVec( targetPos );
  dirVec -= this->getGameObject()->position();

  // Are we further than max visual range from the target.
  float distance = dirVec.length();
  if ( distance > m_visualRange ) {
    return false;
  }
  
  int dist2 = static_cast<int>( distance );
  dirVec.norm();
  dirVec.scale( 8 );


  Vec2D tmpPos( this->getGameObject()->position() );
  for (int i=0; i < dist2; i++) {
    tmpPos += dirVec;
    Vec2D fooV( targetPos );
    fooV -= tmpPos;
    if ( fabs(fooV.vx) < 16 && fabs(fooV.vy) < 16 ) {
      return true;
    }
    if ( Map::bulletCollide(tmpPos) == true ) {
      return false;
    }
  }
  return false;
}



/** Tries to keep the alien away from the walls.
 */
void AIController::avoidWalls()
{
  float a = this->getGameObject()->angle();
  Vec2D pos = this->getGameObject()->position();

  Vec2D refPoint1(0,-32);
  Vec2D refPoint2(0,-32);
  Vec2D refPoint3(0,-32);
  refPoint1.rotate( a );
  refPoint2.rotate( a + 32 );
  refPoint3.rotate( a - 32 );

  refPoint1 += pos;
  refPoint2 += pos;
  refPoint3 += pos;

  bool clockwise = Map::collide( refPoint2 );
  bool cclockwise = Map::collide( refPoint2 );


  if ( Map::collide(refPoint1) ) {
    // There's a wall in front of us.
    if ( clockwise && cclockwise ) {
      // Do full turn
      m_turn = (rand() % 2 == 1) ? -2 : 2;
      m_counter = 40 + (rand() % 20);

    } else if ( clockwise && !cclockwise ) {
      m_turn = -1;
      m_counter = 6 + (rand() % 10);

    } else if ( !clockwise && cclockwise ) {
      m_turn = 1;
      m_counter = 6 + (rand() % 10);

    }
  }
}



/** Sets the random movement.
 */
void AIController::setRandomMovement( int counter )
{
  this->reset();
  m_forward = 1;
  m_turn = (rand() % 3) - 1;
  m_shoot = 0;
  m_counter = counter;
}



/** Are we within the attack range.
 */
bool AIController::withinAttackRange() const
{
  if ( m_pTarget == 0 ) {
    return false;
  }
  
  Vec2D diffV( m_pTarget->position() );
  diffV -= BaseController::m_pObject->position();
  if ( diffV.length() < m_attackRange ) {
    return true;
  }
  return false;
}



/** Are we within the visual range.
 */
bool AIController::withinVisualRange() const
{
  if ( m_pTarget == 0 ) {
    return false;
  }
  
  Vec2D diffV( m_pTarget->position() );
  diffV -= BaseController::m_pObject->position();
  if ( diffV.length() < m_visualRange ) {
    return true;
  }
  return false;
}

float AIController::ApproximateDistance( const Vec2D& aPoint1, const Vec2D& aPoint2 ) const
{
  Vec2D distVec( aPoint2 );
  distVec -= aPoint1;
  return distVec.length();
}

bool AIController::IsSameNode( const Vec2D& aPoint1, const Vec2D& aPoint2 ) const
{
  if ( this->ApproximateDistance(aPoint1, aPoint2) < 32 ) {
    return true;
  }
  return false;
}

void AIController::GetAdjacentNodes( vector<Vec2D>& aAdjacentNodes,
                                    const Vec2D& aCurrentPos ) const
{
  vector<Vec2D> newNodes(8);
  newNodes.at(0) = aCurrentPos + Vec2D(-32,-32);
  newNodes.at(1) = aCurrentPos + Vec2D(  0,-32);
  newNodes.at(2) = aCurrentPos + Vec2D( 32,-32);
  newNodes.at(3) = aCurrentPos + Vec2D(-32,  0);
  newNodes.at(4) = aCurrentPos + Vec2D( 32,  0);
  newNodes.at(5) = aCurrentPos + Vec2D(-32, 32);
  newNodes.at(6) = aCurrentPos + Vec2D(  0, 32);
  newNodes.at(7) = aCurrentPos + Vec2D( 32, 32);
  // Now add the non colliding nodes to the nodetable being returned.
  for ( int i=0; i < newNodes.size(); i++ ) {
    if ( m_pObject->hasMapCollisionAt( newNodes.at(i) ) == false ) {
      aAdjacentNodes.push_back( newNodes.at(i) );
    }
  }
}

float AIController::GetTerrainFactor( const Vec2D& aPos ) const
{
  return 1.0;
}

void AIController::setVisualRange( float aDistance )
{
  m_visualRange = aDistance;
}

void AIController::setAttackDistance( float aDistance )
{
  m_attackRange = aDistance;
}

} // end of namespace
